/* ea_get.c -- this file is part of the SYMLINK Library.

   Copyright (C) 1994, 1995 Ralph Schleicher  */

/* This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU Library General Public License for more details.

   You should have received a copy of the GNU Library General Public
   License along with this library; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  */


#ifdef EMX

#include <sys/ea.h>

#else /* not EMX */

#define INCL_DOSERRORS
#define INCL_DOSFILEMGR

#include <os2.h>

#endif /* not EMX */

#include <errno.h>
#include <string.h>
#include "ansidecl.h"


/* ea_get() reads the extended attribute NAME from a file.  The value
   of the EA will be returned in BUFFER only if BUFFER is huge enough.
   ea_get() returns the size of the EA or -1 if an error occurs.  A
   sequence like

       size = ea_get (handle, name, 0, 0);
       buffer = malloc (size);
       ea_get (handle, name, buffer, size);

   can be used if the size of the EA is not known.  */

int
DEFUN (symlink_ea_get, (handle, name, buffer, size),
int handle AND
const char *name AND
char *buffer AND
size_t size)
{
#ifdef EMX

  struct _ea attr;

  if (!name || (!buffer && size))
    {
      errno = EINVAL;
      return (-1);
    }

  if (_ea_get (&attr, 0, handle, name) == -1)
    return (-1);

  if (size && size >= attr.size && attr.value)
    memcpy (buffer, attr.value, attr.size);

  size = attr.size;
  _ea_free (&attr);

  return (size);

#else /* not EMX */

  int error;
  int length;
  int alloc;
  EAOP2 attr;
  GEA2LIST *get;
  FEA2LIST *full;

  if (!name || (!buffer && size))
    {
      errno = EINVAL;
      return (-1);
    }

  length = strlen (name);

  alloc = sizeof (GEA2LIST) + length + 1;
  get = malloc (alloc);
  if (get == 0)
    return (-1);

  get->cbList = alloc;
  get->list[0].oNextEntryOffset = 0;
  get->list[0].cbName = length;
  strcpy (get->list[0].szName, name);

  alloc = sizeof (FEA2LIST) + length + 1 + size;
  full = malloc (alloc);
  if (full == 0)
    {
      free (get);
      return (-1);
    }

  full->cbList = alloc;

  attr.fpGEA2List = get;
  attr.fpFEA2List = full;

  error = DosQueryFileInfo (handle, 3, &attr, sizeof (attr));
  if (error == 0)
    {
      if (size && size >= full->list[0].cbValue)
	memcpy (buffer, full->list[0].szName + full->list[0].cbName + 1,
	  full->list[0].cbValue);

      length = full->list[0].cbValue;

      free (get);
      free (full);

      return (length);
    }

  free (get);
  free (full);

  switch (error)
    {
    case ERROR_ACCESS_DENIED:

      errno = EACCES;
      return (-1);

    case ERROR_INVALID_HANDLE:
    case ERROR_DIRECT_ACCESS_HANDLE:

      errno = EBADF;
      return (-1);

    case ERROR_BUFFER_OVERFLOW:

      errno = ERANGE;
      return (-1);
    }

  errno = EINVAL;
  return (-1);

#endif /* not EMX */
}
