## -*-Tcl-*-
 # ###################################################################
 #  AEBuild - Functions for building AppleEvents 
 #  			(modernization of appleEvents.tcl)
 # 
 #  FILE: "aebuild.tcl"
 #                                    created: 2/25/98 {7:37:06 PM} 
 #                                last update: 20/4/1999 {12:08:56 am}
 #                                    version: 1.2
 #  Author: Jonathan Guyer
 #  E-mail: <jguyer@his.com>
 #     www: <http://www.his.com/~jguyer/>
 #  
 # Copyright (c) 1998  Jonathan Guyer
 # 
 # This program is free software; you can redistribute it and/or modify
 # it under the terms of the GNU General Public License as published by
 # the Free Software Foundation; either version 2 of the License, or
 # (at your option) any later version.
 # 
 # This program is distributed in the hope that it will be useful,
 # but WITHOUT ANY WARRANTY; without even the implied warranty of
 # MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 # GNU General Public License for more details.
 # 
 # You should have received a copy of the GNU General Public License
 # along with this program; if not, write to the Free Software
 # Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 # See the file "license.terms" for information on usage and 
 # redistribution of this file, and for a DISCLAIMER OF ALL 
 # WARRANTIES.
 #  
 # ###################################################################
 ##

namespace eval aebuild {}

## 
 # -------------------------------------------------------------------------
 # 
 # "aebuild::result" --
 # 
 #  Shorthand routine to get the direct object result of an AEBuild call
 # -------------------------------------------------------------------------
 ##
proc aebuild::result {args} {
	return [aeparse::keywordValue ---- \
		[aeparse::event [eval AEBuild -r $args]] \
	]
}

## 
 # -------------------------------------------------------------------------
 # 
 # "aebuild::fromParsed" --
 # 
 #  Take list of lists generated by aeparse routines and generate the
 #  corresponding AEGizmo string.
 # -------------------------------------------------------------------------
 ##
proc aebuild::fromParsed {args} {
	switch [llength $args] {
		"0" {
			return
		}
		"1" {
			return $args
		}
		"2" {
			if {[llength [info commands "aebuild::[lindex $args 0]"]]} {
				return ["aebuild::[lindex $args 0]" [lindex $args 1]]
			} else {
				return [eval aebuild::coercion [lindex $args 0] [eval aebuild::fromParsed [lindex $args 1]]]
			}
		}
	}
}

proc aebuild::objectProperty {process property object} {
	return [aebuild::result $process core getd ---- \
	  			[propertyObject $property $object]]
}

proc aebuild::coercion {type value} {
	return "'${type}'(${value})"
	# ??? what about coerced records?
	# ??? coerced lists should generate 18 aeBuildSyntaxCoercedList
}

## 
 # -------------------------------------------------------------------------
 # 
 # "aebuild::list" --
 # 
 #  Convert list 'l' to an AE list, i.e., "[l1, l2, l3, ...]".
 #  "-as type" coerces elements to 'type' before joining.  
 #  "-pretyped" 
 # -------------------------------------------------------------------------
 ##
proc aebuild::list {l args} {
	set opts(-as) ""
	getOpts as
	
	if {[string length $opts(-as)] != 0} {
		set out {}
		foreach item $l {
			lappend out [aebuild::$opts(-as) $item]
		}
	} else {
		set out $l
	}
	
	set out [join $out ", "]
	return "\[$out\]"
}

## 
 # -------------------------------------------------------------------------
 # 
 # "aebuild::reco" --
 # 
 #  Convert list 'l' to an AE record, i.e., "{t1:l1, t2:l2, t3:l3, ...}".
 # -------------------------------------------------------------------------
 ##
proc aebuild::reco {items} {
	set out {}
	foreach item $items {
		lappend out "'[lindex $item 0]':[eval aebuild::fromParsed [lindex $item 1]]"
	}
	
	set out [join $out ", "]
	return "\{$out\}"
}

## 
 # -------------------------------------------------------------------------
 # 
 # "aebuild::hexd" --
 # 
 #  Convert 'value' to 'value'.
 #  value's spaces are stripped and it is left-padded with 0 to even digits.
 # -------------------------------------------------------------------------
 ##
proc aebuild::hexd {value} {
	return "[aecoerce::hexd $value]"
}

## 
 # -------------------------------------------------------------------------
 # 
 # "aebuild::bool" --
 # 
 #  Convert 'val' to AE 'bool(val)'.
 # -------------------------------------------------------------------------
 ##
proc aebuild::bool {val} {
    if {($val == 1) || ($val == 0)} {
	return [aebuild::coercion "bool" [aebuild::hexd $val]]
    } else {
	error "${val} is not a valid boolean"
    }
}

## 
 # -------------------------------------------------------------------------
 # 
 # "aebuild::TEXT" --
 #  
 #  Convert 'str' to TEXT.
 #  Curly quotes in 'str' are converted to straight quotes. 
 # -------------------------------------------------------------------------
 ##
proc aebuild::TEXT {str} {
	regsub -all {([])} $str {"} newstr
	return "\$newstr\"
}

## 
 # -------------------------------------------------------------------------
 # 
 # "aebuild::alis" --
 # 
 #  Convert 'path' to an alis(...).
 # -------------------------------------------------------------------------
 ##
proc aebuild::alis {path} {
	return [aebuild::coercion "alis" \
	  [aebuild::hexd [aecoerce::TEXT:alis $path]]]
}

## 
 # -------------------------------------------------------------------------
 # 
 # "aebuild::fss" --
 # 
 #  Convert 'path' to an 'fss '(...).
 # -------------------------------------------------------------------------
 ##
proc aebuild::fss {value} {
	if {([llength $value] == 2) && ([lindex $value 0] == "hexd")} {
		# uncoerced hex data
		return [eval aebuild::coercion "fss " [aebuild::hexd [lindex $value 1]]]
	} else {
		# assume it's a path
		return [eval aebuild::fromParsed \
		  [aeparse::keywordValue ---- \
		    [aeparse::event \
				[AEBuild -r 'MACS' core getd ---- \
					"obj{want:type('cobj'), from:'null'(), \
							[aebuild::name $value] \
					}" \
					rtyp fss \
				] \
		        -noCoerce {{hexd {fss }}} \
		      ] \
		    1 \
		  ] \
		]
	}
}


proc aebuild::name {name} {
	return "form:'name', seld:[aebuild::TEXT $name]"
}

proc aebuild::filename {name} {
	return "obj{want:type('file'), from:'null'(), [aebuild::name $name] } "
}

proc aebuild::winByName {name} {
	return "obj{want:type('cwin'), from:'null'(), [aebuild::name $name] } "
}

proc aebuild::winByPos {absPos} {
	return "obj{want:type('cwin'), from:'null'(), [aebuild::absPos $absPos] } "
}

proc aebuild::lineRange {absPos1 absPos2} {
	set lineObj1 "obj{ want:type('clin'), from:'ccnt'(), [aebuild::absPos $absPos1] }"
	set lineObj2 "obj{ want:type('clin'), from:'ccnt'(), [aebuild::absPos $absPos2] }"
	return "form:'rang', seld:rang{star:$lineObj1, stop:$lineObj2 } "
}

proc aebuild::absPos {posName} {
#
# Use '1' or 'first' to specify first position
# and '-1' or 'last' to specify last position.
#
	if {$posName == "first"} { 
		set posName 1 
	} elseif {$posName == "last"} { 
		set posName -1 
	}
	if {$posName >= -1} {
		return "form:indx, seld:long($posName)"
	} else {
		error "aebuild::absPos: bad argument"
	}
}

#  Utilities  #

## 
 # -------------------------------------------------------------------------
 # 
 # "aebuild::startupDisk" --
 # 
 #  The name of the Startup Disk (as sometimes returned by the Finder)
 # -------------------------------------------------------------------------
 ##
proc aebuild::startupDisk {} {
	return [aebuild::objectProperty 'MACS' pnam \
		  "obj \{want:type(prop), from:'null'(), \
		    form:prop, seld:type(sdsk)\}" \
	]	
}

## 
 # -------------------------------------------------------------------------
 # 
 # "aebuild::OS8userName" --
 # 
 # Get the owner name of the computer from the File Sharing control 
 # panel, 'shcp', which is scriptable as of MacOS 8.x
 #  
 # -------------------------------------------------------------------------
 ##
proc aebuild::OS8userName {} {
	
	# We don't care; just want an error thrown if the File Sharing
	# control panel isn't scriptable
	nameFromAppl shcp
	
	set quitWhenDone [expr ![app::isRunning shcp]]
	
	app::ensureRunning shcp
	
	# tell application "File Sharing" to get owner name
	set userName [aebuild::objectProperty 'shcp' ownn [nullObject]]
	
	# If File Sharing wasn't open before this call, kill it
	if {$quitWhenDone} {
		sendQuitEvent 'shcp'
	} 
	
	return $userName
}

## 
 # -------------------------------------------------------------------------
 # 
 # "aebuild::OS7userName" --
 # 
 # For MacOS 7.x, we use the owner of the preferences folder.
 #  
 # This is not guaranteed to be the same as the Mac's owner, but it's 
 # likely the same and seems preferable to IC's user name, which is almost 
 # never the same.
 #
 # I picked the preference folder because it was easily 
 # specifiable through AppleEvents, because its default ownership 
 # is that of the computer, and because a user would really have to 
 # go out of their way to change it (by either explicitly changing 
 # ownership, or more likely, by clicking 
 # 'Make all currently enclosed folders like this one' 
 # in the startup disk's Sharing window after changing the disk's 
 # ownership. Anyone who does this should be taunted severely.
 # 
 # This will fail if File Sharing is off.
 # -------------------------------------------------------------------------
 ##
proc aebuild::OS7userName {} {
	# tell application "Finder" to get owner of preferences folder
	return [aebuild::objectProperty 'MACS' sown \
		"obj \{want:type(prop), from:'null'(), form:prop, seld:type(pref)\}" \
	]
}

## 
 # -------------------------------------------------------------------------
 # 
 # "aebuild::userName" --
 # 
 #  Return the default user name. The Mac's owner name,
 #  which is in String Resource ID -16096, is inaccesible to Tcl 
 #  (at least until Tcl 8 is implemented).
 #  
 #  Try different mechanisms for determining the user name.
 #  
 # -------------------------------------------------------------------------
 ##
if {[info tclversion] < 8.0} {
proc aebuild::userName {} {
	
    if {[catch {aebuild::OS8userName} userName]} {
	
	# Above failed, probably because the OS doesn't support
	# scriptable File Sharing.
	
	if {[catch {aebuild::OS7userName} userName]} {
	    # Both attempts at a user name failed, so return whatever
	    # Internet Config has
	    
	    set userName [icGetPref RealName]
	}
    }
    
    return $userName
}
} else {
    proc aebuild::userName {} {
	return [text::fromPstring [resource read "STR " -16096]]
    }
    
}
