package com.dickimawbooks.jmakepdfx;

import java.io.*;
import java.util.*;
import java.net.*;
import java.awt.*;
import java.awt.event.*;
import static java.nio.file.StandardCopyOption.*;
import java.nio.file.*;
import java.util.zip.*;
import java.text.*;

import javax.swing.*;
import javax.swing.event.*;
import javax.help.*;

public class Jmakepdfx extends JFrame
   implements ActionListener,MenuListener
{
   public Jmakepdfx()
   {
      super(appName);

      setDefaultCloseOperation(JFrame.DO_NOTHING_ON_CLOSE);

      addWindowListener(new WindowAdapter()
      {
         public void windowClosing(WindowEvent evt)
         {
            quit();
         }
      });


      try
      {
         loadDictionary();
      }
      catch (IOException e)
      {
         error("Unable to load dictionary file:\n"
            +e.getMessage());
      }

      try
      {
         setIconImage(new ImageIcon(getClass().getResource("/icons/logosmall.png")).getImage());
      }
      catch (NullPointerException e)
      {
         debug("Can't find /icons/logosmall.png");
      }

      try
      {
         properties = JpdfxProperties.fetchProperties(this);
      }
      catch (IOException e)
      {
         error(getLabelWithAlt("error.no_properties",
               "Unable to load properties")
            +"\n"+e.getMessage(), e);
         properties = new JpdfxProperties(this);
      }

      toolBar = new JToolBar(properties.getToolBarOrientation());
      getContentPane().add(toolBar, properties.getToolBarPosition());

      initHelp();

      JMenuBar mBar = new JMenuBar();
      setJMenuBar(mBar);

      JMenu fileM = createMenu("file");

      mBar.add(fileM);

      recentFilesListener = new ActionListener()
      {
         public void actionPerformed(ActionEvent evt)
         {
            String action = evt.getActionCommand();

            if (action == null) return;

            try
            {
               int idx = Integer.parseInt(action);

               File file = new File(properties.getRecentFileName(idx));

               pdfFileChooser.setCurrentDirectory(file.getParentFile());

               load(file);
            }
            catch (Exception e)
            {
               error(e);
               setInfo(getLabel("message.failedinfo"));
               clear();
            }
         }
      };

      recentM = createMenu("file", "recent");
      recentM.addMenuListener(this);
      fileM.add(recentM);

      clearRecentItem = createMenuItem("file.recent", "clearrecent");

      openItem = createMenuButtonItem("file", "open",
         KeyStroke.getKeyStroke(KeyEvent.VK_O, InputEvent.CTRL_MASK),
         "general/Open24", fileM);

      fileM.add(createMenuItem("file", "quit",
        KeyStroke.getKeyStroke(KeyEvent.VK_Q, InputEvent.CTRL_MASK)));

      JMenu toolsM = createMenu("tools");
      mBar.add(toolsM);

      convertItem = createMenuItem("tools", "convert");
      toolsM.add(convertItem);

      abortItem = createMenuButtonItem("tools", "abort",
         KeyStroke.getKeyStroke(KeyEvent.VK_ESCAPE, 0),
         "general/Stop24", toolsM);

      abortItem.setEnabled(false);

      JMenu settingsM = createMenu("settings");

      mBar.add(settingsM);

      MenuButtonItem editSettingsItem = createMenuButtonItem("settings",
       "editsettings", KeyStroke.getKeyStroke(KeyEvent.VK_P, InputEvent.CTRL_MASK),
        "general/Preferences24", settingsM);

      JMenu helpM = createMenu("help");
      mBar.add(helpM);

      helpM.add(createMenuItem("help", "about"));

      helpM.add(createMenuItem("help", "license"));

      createMenuButtonItem("help", "manual",
         KeyStroke.getKeyStroke(KeyEvent.VK_F1, 0),
         "general/Help24", csh, helpM);

      pdfFileChooser = new JFileChooser(new File(properties.getDefaultDirectory()));
      pdfFileChooser.setFileFilter(new PdfFileFilter());
      pdfFileChooser.setAcceptAllFileFilterUsed(false);

      propertiesDialog = new PropertiesDialog(this);

      appSelector = new JpdfxAppSelector(this);
      iccSelector = new IccSelector(this);

      JPanel panel = new JPanel(new BorderLayout());

      Box mainPanel = Box.createVerticalBox();

      mainPanel.add(new Box.Filler(
       new Dimension(0, 0), 
       new Dimension(1, 20),
       new Dimension(Integer.MAX_VALUE, 20)));

      Box inputBox = Box.createHorizontalBox();
      mainPanel.add(inputBox);

      inputLabel = new JLabel(getLabel("main", "input"));
      inputLabel.setDisplayedMnemonic(getMnemonic("main", "input"));
      inputBox.add(inputLabel);

      inputField = new JTextField(40);
      inputLabel.setLabelFor(inputField);
      inputBox.add(inputField);
      inputField.setEditable(false);

      inputButton = new JButton("...");
      inputButton.setActionCommand("open");
      inputButton.addActionListener(this);
      inputBox.add(inputButton);

      Box outputBox = Box.createHorizontalBox();
      mainPanel.add(outputBox);

      outputLabel = new JLabel(getLabel("main", "output"));
      outputLabel.setDisplayedMnemonic(getMnemonic("main", "output"));
      outputBox.add(outputLabel);

      outputField = new JTextField(40);
      outputLabel.setLabelFor(outputField);
      outputBox.add(outputField);

      outputButton = new JButton("...");
      outputButton.setActionCommand("setoutput");
      outputButton.addActionListener(this);
      outputBox.add(outputButton);

      Dimension dim = outputLabel.getPreferredSize();

      int prefW = Math.max(dim.width, (int)inputLabel.getPreferredSize().getWidth());
      dim.width = prefW;

      int prefH = dim.height+4;

      inputLabel.setPreferredSize(dim);
      outputLabel.setPreferredSize(dim);

      dim = inputField.getMaximumSize();
      dim.height = prefH;

      inputField.setMaximumSize(dim);

      dim = outputField.getMaximumSize();
      dim.height = prefH;

      outputField.setMaximumSize(dim);
      mainPanel.add(new Box.Filler(
       new Dimension(0, 0), 
       new Dimension(1, 20),
       new Dimension(Integer.MAX_VALUE, 20)));

      infoBox = Box.createVerticalBox();
      mainPanel.add(infoBox);

      infoBox.setBorder(BorderFactory.createTitledBorder(
       BorderFactory.createEtchedBorder(), 
       getLabel("main", "info")));

      Box box = Box.createHorizontalBox();
      infoBox.add(box);

      authorLabel = new JLabel(getLabel("main", "author"));
      authorLabel.setDisplayedMnemonic(getMnemonic("main", "author"));
      box.add(authorLabel);

      authorField = new JTextField(20);
      authorLabel.setLabelFor(authorField);
      box.add(authorField);

      box = Box.createHorizontalBox();
      infoBox.add(box);

      titleLabel = new JLabel(getLabel("main", "title"));
      titleLabel.setDisplayedMnemonic(getMnemonic("main", "title"));
      box.add(titleLabel);

      titleField = new JTextField(20);
      titleLabel.setLabelFor(titleField);
      box.add(titleField);

      box = Box.createHorizontalBox();
      infoBox.add(box);

      pageCountLabel = new JLabel(getLabel("main", "pagecount"));
      box.add(pageCountLabel);

      pageCountField = new JTextField(20);
      pageCountField.setEditable(false);
      box.add(pageCountField);

      fileSizeLabel = new JLabel(getLabel("main", "filesize"));
      box.add(fileSizeLabel);

      fileSizeField = new JTextField(20);
      fileSizeField.setEditable(false);
      box.add(fileSizeField);

      dim = authorLabel.getPreferredSize();

      dim.width = (int)Math.max(
       Math.max(dim.width,
       fileSizeLabel.getPreferredSize().getWidth()), 
       (int)Math.max(titleLabel.getPreferredSize().getWidth(),
                     pageCountLabel.getPreferredSize().getWidth()));

      authorLabel.setPreferredSize(dim);
      titleLabel.setPreferredSize(dim);
      pageCountLabel.setPreferredSize(dim);
      fileSizeLabel.setPreferredSize(dim);

      mainPanel.add(new Box.Filler(
       new Dimension(0, 0), 
       new Dimension(1, 20),
       new Dimension(Integer.MAX_VALUE, 20)));

      Box selectorBox = Box.createHorizontalBox();
      mainPanel.add(selectorBox);

      selectorBox.add(Box.createHorizontalGlue());

      profileLabel = new JLabel(getLabel("main", "profile"));
      selectorBox.add(profileLabel);

      ButtonGroup bg = new ButtonGroup();

      boolean isGrey = properties.isGrayProfile();

      greyButton = new JRadioButton(getLabel("main", "grey"), isGrey);
      greyButton.setMnemonic(getMnemonic("main", "grey"));
      bg.add(greyButton);
      selectorBox.add(greyButton);

      cmykButton = new JRadioButton(getLabel("main", "cmyk"), !isGrey);
      cmykButton.setMnemonic(getMnemonic("main", "cmyk"));
      bg.add(cmykButton);
      selectorBox.add(cmykButton);

      useIccBox = new JCheckBox(getLabel("main", "useicc"), properties.isUseICC());
      useIccBox.setMnemonic(getMnemonic("main", "useicc"));
      selectorBox.add(useIccBox);

      convertButton = new JButton(getLabel("main", "convert"));
      convertButton.setMnemonic(getMnemonic("main", "convert"));
      convertButton.setActionCommand("convert");
      convertButton.addActionListener(this);
      selectorBox.add(convertButton);

      selectorBox.add(Box.createHorizontalGlue());

      mainPanel.add(new Box.Filler(
       new Dimension(0, 0), 
       new Dimension(1, 20),
       new Dimension(Integer.MAX_VALUE, Integer.MAX_VALUE)));

      panel.add(new JScrollPane(mainPanel), "Center");

      messageArea = new MessageArea(getLabel("message.init"));

      panel.add(messageArea, "South");

      getContentPane().add(panel);

      pack();

      currentProcessListeners = new Vector<ProcessListener>();
      currentWorkerThreads = new Vector<Thread>();

      clear();

      setLocationRelativeTo(null);
      setVisible(true);
   }

   public void menuSelected(MenuEvent evt)
   {
      Object source = evt.getSource();

      if (source == recentM)
      {
         properties.setRecentFiles(recentM, recentFilesListener);

         recentM.addSeparator();
         recentM.add(clearRecentItem);
      }
   }

   public void menuDeselected(MenuEvent evt)
   {
   }

   public void menuCanceled(MenuEvent evt)
   {
   }

   public void actionPerformed(ActionEvent evt)
   {
      String action = evt.getActionCommand();

      if (action == null) return;

      if (action.equals("open"))
      {
         if (pdfFileChooser.showOpenDialog(this)
           == JFileChooser.APPROVE_OPTION)
         {
            try
            {
               load(pdfFileChooser.getSelectedFile());
            }
            catch (ProcessFailedException e)
            {
               error(e.getMessage());
               setInfo(getLabel("message.failedinfo"));
               clear();
            }
            catch (Exception e)
            {
               error(e);
               setInfo(getLabel("message.failedinfo"));
               clear();
            }
         }
      }
      else if (action.equals("clearrecent"))
      {
         properties.clearRecentList();
      }
      else if (action.equals("setoutput"))
      {
         File file;

         String fileName = outputField.getText();

         if (fileName.equals(""))
         {
            String base = currentFile.getName();

            int idx = base.lastIndexOf(".");

            if (idx > -1)
            {
               base = base.substring(0, idx);
            }

            file = new File(currentFile.getParentFile(), 
               base+"-pdfx.pdf");
         }
         else
         {
            file = new File(fileName);
         }

         pdfFileChooser.setCurrentDirectory(file.getParentFile());
         pdfFileChooser.setSelectedFile(file);

         if (pdfFileChooser.showDialog(this, getLabel("button", "select"))
           == JFileChooser.APPROVE_OPTION)
         {
            setOutput(pdfFileChooser.getSelectedFile());
         }
      }
      else if (action.equals("quit"))
      {
         quit();
      }
      else if (action.equals("abort"))
      {
         abort();
      }
      else if (action.equals("convert"))
      {
         Thread thread = new ConvertThread(this);
         startWorkerThread(thread);
      }
      else if (action.equals("editsettings"))
      {
         propertiesDialog.display();
      }
      else if (action.equals("about"))
      {
         String[] str;

         String translator = dictionary.getProperty("about.translator_info");

         if (translator == null || translator.equals(""))
         {
            str = new String[]
            {
               appName,
               getLabelWithValue("about.version", appVersion),
               getLabelWithValues("about.copyright", "Nicola L. C. Talbot",
                   appDate),
               "http://www.dickimaw-books.com/"
            };
         }
         else
         {
            str = new String[]
            {
               appName,
               getLabelWithValue("about.version", appVersion),
               getLabelWithValues("about.copyright", "Nicola L. C. Talbot",
                  appDate),
               "http://www.dickimaw-books.com/",
               translator
            };
         }

         JOptionPane.showMessageDialog(this, str,
            getLabelWithValue("about.title", appName), JOptionPane.PLAIN_MESSAGE);
      }
      else if (action.equals("license"))
      {
         if (licenceDialog == null)
         {
            try
            {
               licenceDialog = new LicenceDialog(this);
            }
            catch (IOException e)
            {
               error(e);
               licenceDialog = null;
            }
         }

         if (licenceDialog != null)
         {
            licenceDialog.setVisible(true);
         }
      }
   }

   public void quit()
   {
      if (currentWorkerThreads.size() > 0)
      {
         if (JOptionPane.showConfirmDialog(this, 
             getLabel("message.confirm.quit"),
             getLabel("message.confirm"),
             JOptionPane.YES_NO_OPTION)
          != JOptionPane.YES_OPTION)
         {
            return;
         }

         for (Thread thread : currentWorkerThreads)
         {
            debug("Interrupting "+thread+" "+thread.getClass().toString());
            thread.interrupt();
         }

         for (ProcessListener listener : currentProcessListeners)
         {
            listener.terminateProcess();
         }

         abortItem.setEnabled(false);

         currentWorkerThreads.clear();
         currentProcessListeners.clear();
      }

      try
      {
         properties.setToolBarPosition((String)
            ((BorderLayout)getContentPane().getLayout()).getConstraints(toolBar));
         properties.setToolBarOrientation(toolBar.getOrientation());

         properties.setProfile(greyButton.isSelected() ? "gray" : "cmyk");
         properties.setUseICC(useIccBox.isSelected());

         properties.save();
      }
      catch (Exception e)
      {
         error(getLabel("error.propsave_failed")+"\n"+e.getMessage(), e);
      }

      System.exit(0);
   }

   public void setOutput(File outputFile)
   {
      outputField.setText(outputFile.getAbsolutePath());
   }

   public String getOutputName()
   {
      return outputField.getText();
   }

   public void clear()
   {
      inputField.setText("");
      outputField.setText("");
      authorField.setText("");
      titleField.setText("");
      pageCountField.setText("");
      fileSizeField.setText("");
      disableFunctions();
      abortItem.setEnabled(false);
   }

   public void viewPDF(String file)
   {
      String viewer = properties.getPdfViewer();

      if (viewer == null || viewer.equals(""))
      {
         viewer = appSelector.fetchApplicationPath(
             getLabelWithValue("properties.query.location.viewer", "PDF"))
                .getAbsolutePath();

         if (viewer == null || viewer.equals(""))
         {
            error(getLabelWithValue("error.load_type_failed", "PDF"));
            return;
         }

         properties.setPdfViewer(viewer);
      }

      String[] cmd = new String[]{viewer, file};

      try
      {
         Process p = execCommand(cmd, false);
      }
      catch (Exception e)
      {
         error(getLabelWithValue("error.exec_failed", cmd[0]+" "+cmd[1])
            +"\n"+e.getMessage());
      }
   }

   public void load(File pdfFile)
      throws IOException,InterruptedException,ProcessFailedException
   {
      setWaitCursor();

      try
      {
         getPdfInfo(pdfFile);

         currentFile = pdfFile;
         setTitle(appName+" - "+currentFile.getName());
         properties.addRecentFile(currentFile.getAbsolutePath());

         resetFunctions();

         String base = currentFile.getName();

         int idx = base.lastIndexOf(".");

         if (idx > -1)
         {
            base = base.substring(0, idx)+"-pdfx";
         }
         else
         {
            base += "-pdfx";
         }

         outputField.setText((new File(currentFile.getParentFile(), base+".pdf")).getAbsolutePath());
         inputField.setText(currentFile.getAbsolutePath());
         revalidate();

         setInfo(getLabel("message", "file_loaded"));
      }
      finally
      {
         setDefaultCursor();
      }
   }

   private JMenu createMenu(String label)
   {
      return createMenu(null, label);
   }

   private JMenu createMenu(String parentLabel, String label)
   {
      JMenu menu = new JMenu(getLabel(parentLabel, label));
      menu.setMnemonic(getMnemonic(parentLabel, label));

      return menu;
   }

   private JMenuItem createMenuItem(String parentLabel, String label)
   {
      return createMenuItem(parentLabel, label, null, null, this);
   }

   private JMenuItem createMenuItem(String parentLabel, String label, KeyStroke keyStroke)
   {
      return createMenuItem(parentLabel, label, keyStroke, null, this);
   }

   private JMenuItem createMenuItem(String parentLabel, String label, String tooltip)
   {
      return createMenuItem(parentLabel, label, null, tooltip, this);
   }

   private JMenuItem createMenuItem(String parentLabel, String label,
      KeyStroke keyStroke, String tooltip)
   {
      return createMenuItem(parentLabel, label, keyStroke, tooltip, this);
   }

   private JMenuItem createMenuItem(String parentLabel, String label,
      KeyStroke keyStroke, String tooltip, ActionListener listener)
   {
      JMenuItem item = new JMenuItem(getLabel(parentLabel, label));
      item.setMnemonic(getMnemonic(parentLabel, label));
      item.setActionCommand(label);

      if (listener != null)
      {
         item.addActionListener(listener);
      }

      if (keyStroke != null)
      {
         item.setAccelerator(keyStroke);
      }

      if (tooltip != null)
      {
         item.setToolTipText(tooltip);
      }

      return item;
   }

   private MenuButtonItem createMenuButtonItem(String parentLabel, String label,
      KeyStroke keyStroke, String imageName, JMenu menu)
   {
      return createMenuButtonItem(parentLabel, label, keyStroke, imageName, this, menu);
   }

   private MenuButtonItem createMenuButtonItem(String parentLabel, String label,
      KeyStroke keyStroke, String imageName, ActionListener listener, JMenu menu)
   {
      return createMenuButtonItem(parentLabel, label, keyStroke, imageName, listener,
        menu, toolBar);
   }

   public static MenuButtonItem createMenuButtonItem(String parentLabel, String label,
      KeyStroke keyStroke, String imageName, ActionListener listener, JMenu menu,
      JComponent toolBarComp)
   {
      String tooltip = dictionary.getProperty(parentLabel+"."+label+".tooltip");
      String altText = dictionary.getProperty(parentLabel+"."+label+".altText");

      String imgLocation;

      if (imageName.endsWith(".png"))
      {
         imgLocation = "icons/"+imageName;
      }
      else
      {
         imgLocation = "/toolbarButtonGraphics/"+imageName+".gif";
      }

      URL imageURL = Jmakepdfx.class.getResource(imgLocation);

      MenuButtonItem item = new MenuButtonItem(getLabel(parentLabel, label),
        getMnemonic(parentLabel, label), imageURL, label,
        keyStroke, listener, tooltip, altText);

      toolBarComp.add(item.getButton());
      menu.add(item.getItem());

      return item;
   }

   public static String getLabelWithAlt(String label, String alt)
   {
      if (dictionary == null) return alt;

      String prop = dictionary.getProperty(label);

      if (prop == null)
      {
         return alt;
      }

      return prop;
   }

   public static String getLabel(String label)
   {
      return getLabel(null, label);
   }

   public static String getLabel(String parent, String label)
   {
      if (parent != null)
      {
         label = parent+"."+label;
      }

      String prop = dictionary.getProperty(label);

      if (prop == null)
      {
         System.err.println("No such dictionary property '"+label+"'");
         return "?"+label+"?";
      }

      return prop;
   }

   public static char getMnemonic(String label)
   {
      return getMnemonic(null, label);
   }

   public static char getMnemonic(String parent, String label)
   {
      String prop = getLabel(parent, label+".mnemonic");

      if (prop.equals(""))
      {
         System.err.println("Empty dictionary property '"+prop+"'");
         return label.charAt(0);
      }

      return prop.charAt(0);
   }

   public static int getMnemonicInt(String label)
   {
      return getMnemonicInt(null, label);
   }

   public static int getMnemonicInt(String parent, String label)
   {
      String prop = getLabel(parent, label+".mnemonic");

      if (prop.equals(""))
      {
         System.err.println("Empty dictionary property '"+prop+"'");
         return label.codePointAt(0);
      }

      return prop.codePointAt(0);
   }

   public static String getLabelWithValue(String label, String value)
   {
      String prop = getLabel(label);

      return prop.replaceAll("\\$1", value.replaceAll("\\\\", "\\\\\\\\"));
   }

   public static String getLabelWithValue(String label, int value)
   {
      return getLabelWithValue(label, ""+value);
   }

   public static String getLabelWithValues(String label, String value1,
      String value2)
   {
      String prop = getLabel(label);

      return prop.replaceAll("\\$1", value1.replaceAll("\\\\", "\\\\\\\\"))
         .replaceAll("\\$2", value2.replaceAll("\\\\", "\\\\\\\\"));
   }

   public static String getLabelWithValues(String label, String[] values)
   {
      String prop = getLabel(label);

      for (int i = 0; i < values.length; i++)
      {
         prop = prop.replaceAll("\\$"+(i+1), values[i].replaceAll("\\\\", "\\\\\\\\"));
      }

      return prop;
   }

   public void setInfo(String message)
   {
      messageArea.update(message);
   }

   public void setInfoAndDebug(String message)
   {
      debug(message);
      setInfo(message);
   }

   public void setWaitCursor()
   {
      Thread thread = new Thread()
      {
         public void run()
         {
            setCursor(Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR));
         }
      };
      thread.start();
      thread = null;
   }

   public void setDefaultCursor()
   {
      Thread thread = new Thread()
      {
         public void run()
         {
            setCursor(Cursor.getPredefinedCursor(Cursor.DEFAULT_CURSOR));
         }
      };
      thread.start();
      thread = null;
   }

   public File fetchICCFile()
     throws IOException,InterruptedException
   {
      URL url = new URL("http://www.colormanagement.org/download_files/ISOcoated_v2_basICC.zip");

      File targetDir = pdfFileChooser.getCurrentDirectory();

         // Confirm destination directory

      JFileChooser dirChooser = new JFileChooser(
         targetDir == null ? null : targetDir.getParentFile());
      dirChooser.setFileSelectionMode(JFileChooser.DIRECTORIES_ONLY);

      if (targetDir != null)
      {
         dirChooser.setSelectedFile(targetDir);
      }

      if (dirChooser.showDialog(this, getLabel("button.select"))
          != JFileChooser.APPROVE_OPTION)
      {
         return null;
      }

      targetDir = dirChooser.getSelectedFile();

      File file = new File(targetDir, "FOGRA39L.icc");

      unpackZipEntry(url, targetDir, ".icc", file.getName());

      return file;
   }

   public String getCurrentDirectory()
   {
      return pdfFileChooser.getCurrentDirectory().getAbsolutePath();
   }

   public JpdfxProperties getProperties()
   {
      return properties;
   }

   public boolean toPdfX()
      throws IOException,InterruptedException
   {
      String iccFileName = null;
      String iccName = null;

      boolean useIcc = useIccBox.isSelected();

      if (useIcc)
      {
         String icc = properties.getICCFile();

         File iccFile;

         if (icc == null || icc.equals(""))
         {
            iccFile = iccSelector.fetchPath();

            if (iccFile == null)
            {
               // user cancelled

               throw (new FileNotFoundException(getLabel("error.no_icc")));
            }
         }
         else
         {
            iccFile = new File(icc);
         }

         iccFileName = toFileName(iccFile);

         iccName = iccFile.getName().replaceAll("([\\(\\)])", "\\\\1");

         int idx = iccName.toLowerCase().lastIndexOf(".icc");

         if (idx > 0)
         {
            iccName = iccName.substring(0, idx);
         }
      }

      String gsApp = getGSApp();

      String outputFileName = outputField.getText();

      if (outputFileName == null || outputFileName.equals(""))
      {
         throw (new FileNotFoundException(getLabel("error.no_outputfile")));
      }

      File outputFile = new File(outputFileName);

      if (outputFile.isDirectory())
      {
         throw (new IOException(getLabelWithValue("error.output_is_dir",
           outputFileName)));
      }

      String fileName = currentFile.getAbsolutePath();

      File inputFile = new File(fileName);

      if (outputFile.getCanonicalPath().equals(inputFile.getCanonicalPath()))
      {
         throw new IOException(getLabel("error.io.outin"));
      }

      if (outputFile.exists())
      {
         if (JOptionPane.showConfirmDialog(this, 
             getLabelWithValue("message.confirm.overwrite", outputFile.getAbsolutePath()),
             getLabel("message.confirm"),
             JOptionPane.YES_NO_OPTION)
           != JOptionPane.YES_OPTION)
         {
            setInfo("message.failed");
            return false;
         }
      }

      checkForInterrupt();

      // create def file

      File dir = currentFile.getParentFile();

      File defFile = File.createTempFile("jmakepdfx", ".ps");
      defFile.deleteOnExit();

      PrintWriter out = null;

      setInfo(getLabelWithValue("message.writing", defFile.getAbsolutePath()));

      try
      {
         out = new PrintWriter(new FileWriter(defFile));

         out.println("systemdict /ProcessColorModel known {");
         out.println("systemdict /ProcessColorModel get dup /DeviceGray ne exch /DeviceCMYK ne and } {");
         out.println("  true");
         out.println("} ifelse");
         out.println("{ (ERROR: ProcessColorModel must be /DeviceGray or DeviceCMYK.)=");
         out.println("  /ProcessColorModel cvx /rangecheck signalerror");
         out.println("} if");

         out.println();
         out.println("% Define entries to the document Info dictionary :");
         out.println();

         if (useIcc)
         {
            out.println("/ICCProfile ("+iccFileName+") def");
            out.println();
         }

         out.println("[ /GTS_PDFXVersion (PDF/X-3:2002)");

         String title = titleField.getText();

         if (title == null)
         {
            title = "";
         }
         else
         {
            title = title.replaceAll("(\\\\)", "\\\\\\1");
            title = title.replaceAll("([\\(\\)])", "\\\\1");
         }

         String author = authorField.getText();

         if (author == null)
         {
            author = "";
         }
         else
         {
            author = author.replaceAll("(\\\\)", "\\\\\\1");
            author = author.replaceAll("([\\(\\)])", "\\\\1");
         }

         String device = (cmykButton.isSelected() ? "CMYK" : "Gray");

         out.println("  /Trapped /False");
         out.println("  /DOCINFO pdfmark");

         if (useIcc)
         {
            out.println();
            out.println("% Define an ICC profile :");
            out.println();

            out.println("currentdict /ICCProfile known {");
            out.println("  [/_objdef {icc_PDFX} /type /stream /OBJ pdfmark");
            out.print("  [{icc_PDFX} <</N systemdict /ProcessColorModel get ");
            out.print("/Device" + device);
            out.println(" eq {1} {4} ifelse >> /PUT pdfmark");
            out.println("  [{icc_PDFX} ICCProfile (r) file /PUT pdfmark");
            out.println("} if");
         }

         out.println();
         out.println("% Define the output intent dictionary :");
         out.println();

         out.println("[/_objdef {OutputIntent_PDFX} /type /dict /OBJ pdfmark");
         out.println("[{OutputIntent_PDFX} <<");
         out.println("  /Type /OutputIntent");
         out.println("  /S /GTS_PDFX");
         out.println("  /OutputCondition (Commercial and speciality printing)");

         if (useIcc)
         {
            out.println("  /OutputConditionIdentifier ("+iccName+")");
            out.println("  /RegistryName (http://www.color.org)");
            out.println("  currentdict /ICCProfile known {");
            out.println("    /DestOutputProfile {icc_PDFX}");
            out.println("  } if");
         }

         out.println(">> /PUT pdfmark");
         out.println("[{Catalog} <</OutputIntents [ {OutputIntent_PDFX} ]>> /PUT pdfmark");

         // convert file names into platform independent paths that
         // ghostscript will be happy with

         String defFileName = toFileName(defFile);
         fileName = toFileName(inputFile);
         outputFileName = toFileName(outputFile);

         checkForInterrupt();

         String[] cmd = 
            new String[]
            {
               gsApp,
               "-dPDFX",
               "-dBATCH",
               "-dNOPAUSE",
               "-dNOOUTERSAVE",
               "-sDEVICE=pdfwrite",
               "-sColorConversionStrategy="+device,
               "-dProcessColorModel=/Device"+device,
               "-dPDFSETTINGS=/prepress",
               "-sOutputFile="+outputFileName+"",
               defFileName,
               "-c",
               "("+fileName+") run " 
                 +"/pdfmark where {pop} {userdict /pdfmark /cleartomark load put} ifelse "
                 +"[/Title ("+title+") /Author ("+author+") /DOCINFO pdfmark"
            };

         checkForInterrupt();

         PdfxProcessListener listener = new PdfxProcessListener(this);

         int exitCode = execCommandAndWaitFor(cmd, listener);

         checkForInterrupt();

         if (exitCode != 0)
         {
            String mess = getLabelWithValue("error.exec_failed", listener.getErrors());
            error(mess);
            System.err.println(mess);

            return false;
         }

         checkForInterrupt();

         String mess = listener.getMessage();

         if (mess != null)
         {
            warning(mess);
         }
      }
      finally
      {
         if (out != null)
         {
            out.close();

            out = null;
         }
      } 

      checkForInterrupt();

      setInfo(getLabel("message.completed"));

      JOptionPane.showMessageDialog(this, getLabel("message.done"));

      return true;
   }

   public void processFinished()
   {
      setDefaultCursor();
      resetFunctions();
   }

//http://www.forward.com.au/javaProgramming/HowToStopAThread.html
   public static synchronized void checkForInterrupt()
     throws InterruptedException
   {
      Thread.yield();

      if (Thread.currentThread().isInterrupted())
      {
         throw new CancelledException();
      }
   }

   public synchronized boolean abort()
   {
      if (JOptionPane.showConfirmDialog(null, 
         getLabel("message.confirm.abort"),
         getLabel("message.confirm"),
         JOptionPane.YES_NO_OPTION) == JOptionPane.YES_OPTION)
      {
         for (Thread thread : currentWorkerThreads)
         {
            debug("Interrupting "+thread+" "+thread.getClass().toString());
            thread.interrupt();
         }

         for (ProcessListener listener : currentProcessListeners)
         {
            listener.terminateProcess();
         }

         abortItem.setEnabled(false);

         currentWorkerThreads.clear();
         currentProcessListeners.clear();

         return true;
      }

      return false;
   }

   public synchronized void addProcessListener(ProcessListener listener)
   {
      currentProcessListeners.add(listener);
   }

   public synchronized void removeProcessListener(ProcessListener listener)
   {
      currentProcessListeners.remove(listener);
   }

   public synchronized void startWorkerThread(Thread thread)
   {
      debug("adding thread to current list");
      currentWorkerThreads.add(thread);
      debug("starting thread");
      thread.start();
      abortItem.setEnabled(true);
   }

   public synchronized void workerThreadFinished(Thread thread)
   {
      currentWorkerThreads.remove(thread);

      updateAbortItem();
   }

   private synchronized void updateAbortItem()
   {
      abortItem.setEnabled(currentWorkerThreads.size() > 0 || 
         currentProcessListeners.size() > 0);
   }

   public void error(String message)
   {
      JpdfxResources.error(this, message);
   }

   public void error(String message, Exception e)
   {
      JpdfxResources.error(this, message, e);
   }

   public void error(Exception e)
   {
      JpdfxResources.error(this, e);
   }

   public void warning(String message)
   {
      JpdfxResources.warning(this, message);
   }

   private void initHelp()
   {
      if (mainHelpBroker == null)
      {
         HelpSet mainHelpSet = null;

         String resource = "jmakepdfx";

         String helpsetLocation = "/resources/helpsets/"+resource;

         String lang = Locale.getDefault().getLanguage();

         try
         {
            URL hsURL = getClass().getResource(helpsetLocation+"-"+lang+"/"+resource+".hs");

            if (hsURL == null && !lang.equals("en"))
            {
               hsURL = getClass().getResource(helpsetLocation+"-en/"+resource+".hs");
            }

            mainHelpSet = new HelpSet(null, hsURL);
         }
         catch (Exception e)
         {
            error("/resources/helpsets/"+resource+".hs\n"+
              getLabel("error.io.helpset")+":\n" +e.getMessage());
         }

         if (mainHelpSet != null)
         {
            mainHelpBroker = mainHelpSet.createHelpBroker();
         }

         if (mainHelpBroker != null)
         {
            csh = new CSH.DisplayHelpFromSource(mainHelpBroker);
         }
      }
   }

   public void enableHelpOnButton(JComponent comp, String id)
   {
      if (mainHelpBroker != null)
      {
         try
         {
            mainHelpBroker.enableHelpOnButton(comp, id,
               mainHelpBroker.getHelpSet());
         }
         catch (BadIDException e)
         {
            error(e);
         }
      }
      else
      {
         System.err.println("Can't enable help on button (id="+id
           +"): null help broker");
      }
   }

   private void loadDictionary()
      throws IOException
   {
      Locale locale = Locale.getDefault();

      String lang = locale.getLanguage();

      String resource = "jmakepdfx";

      InputStream in = 
         getClass().getResourceAsStream("/resources/dictionaries/"+resource+"-"+lang+".prop");

      if (in == null && !lang.equals("en"))
      {
         in = getClass().getResourceAsStream("/resources/dictionaries/"+resource+"-en.prop");
      }

      if (in == null)
      {
         throw new FileNotFoundException
            ("Can't find dictionary resource file /resources/dictionaries/"+resource+"-en");
      }

      BufferedReader reader = new BufferedReader(new InputStreamReader(in));

      dictionary = new Properties();
      dictionary.load(reader);

      reader.close();

      in.close();
   }

   // Fetch and unpack zip file
   // Code from
   // http://www.java2s.com/Tutorial/Java/0180__File/UnpackanarchivefromaURL.htm

   public void unpackArchive(URL url, File targetDir)
     throws IOException,InterruptedException
   {
      if (!(targetDir.exists() || targetDir.mkdirs()))
      {
         throw new IOException(getLabelWithValue("error.io.cant_mkdir",
           targetDir.getAbsolutePath()));
      } 

      setInfo(getLabelWithValue("message.downloading", url.toString()));

      InputStream in = new BufferedInputStream(url.openStream(), 1024);

      Path zip = Files.createTempFile("archive", ".zip");

      Files.copy(in, zip, StandardCopyOption.REPLACE_EXISTING);

      File zipFile = zip.toFile();
      zipFile.deleteOnExit();

      unpackArchive(zipFile, targetDir);
   }

// Unpack entries ending with (case-insensitive) suffix to target
// directory
   public void unpackZipEntries(URL url, File targetDir, String suffix)
     throws IOException,InterruptedException
   {
      if (!(targetDir.exists() || targetDir.mkdirs()))
      {
         throw new IOException(getLabelWithValue("error.io.cant_mkdir",
           targetDir.getAbsolutePath()));
      } 

      suffix = suffix.toLowerCase();

      setInfo(getLabelWithValue("message.downloading", url.toString()));

      InputStream in = new BufferedInputStream(url.openStream(), 1024);

      Path zip = Files.createTempFile("archive", ".zip");
      zip.toFile().deleteOnExit();

      Files.copy(in, zip, StandardCopyOption.REPLACE_EXISTING);

      ZipFile zipFile = new ZipFile(zip.toFile());

      boolean overWriteAll = false;

      for (Enumeration<? extends ZipEntry> entries = zipFile.entries(); entries.hasMoreElements();)
      {
         ZipEntry entry = entries.nextElement();

         String name = entry.getName();

         if (name.toLowerCase().endsWith(suffix))
         {
            overWriteAll = unpackFile(targetDir, zipFile, entry, overWriteAll);
         }
      }
   }

   public void unpackZipEntry(URL url, File targetDir, String suffix, String fileName)
     throws IOException,InterruptedException
   {
      if (!(targetDir.exists() || targetDir.mkdirs()))
      {
         throw new IOException(getLabelWithValue("error.io.cant_mkdir",
           targetDir.getAbsolutePath()));
      } 

      suffix = suffix.toLowerCase();

      setInfo(getLabelWithValue("message.downloading", url.toString()));

      InputStream in = new BufferedInputStream(url.openStream(), 1024);

      Path zip = Files.createTempFile("archive", ".zip");
      zip.toFile().deleteOnExit();

      Files.copy(in, zip, StandardCopyOption.REPLACE_EXISTING);

      ZipFile zipFile = new ZipFile(zip.toFile());

      boolean overWriteAll = false;

      for (Enumeration<? extends ZipEntry> entries = zipFile.entries(); entries.hasMoreElements();)
      {
         ZipEntry entry = entries.nextElement();

         String name = entry.getName();

         if (name.toLowerCase().endsWith(suffix))
         {
            overWriteAll = unpackFile(targetDir, zipFile, entry, fileName, overWriteAll);
            break;
         }
      }
   }

   public void unpackArchive(File theFile, File targetDir)
      throws IOException,InterruptedException
   {
      if (!theFile.exists())
      {
         throw new IOException(getLabelWithValue("error.file_not_found",
            theFile.getAbsolutePath()));
      }

      if (!(targetDir.exists() || targetDir.mkdirs()))
      {
         throw new IOException(getLabelWithValue("error.io.cant_mkdir",
           targetDir.getAbsolutePath()));
      }

      boolean overWriteAll = false;

      ZipFile zipFile = new ZipFile(theFile);

      for (Enumeration<? extends ZipEntry> entries = zipFile.entries(); entries.hasMoreElements();)
      {
         ZipEntry entry = entries.nextElement();

         overWriteAll = unpackFile(targetDir, zipFile, entry, overWriteAll);
      }

      zipFile.close();
   }

   public boolean unpackFile(File targetDir, ZipFile zipFile, ZipEntry entry, boolean overWriteAll)
     throws IOException,InterruptedException
   {
      return unpackFile(targetDir, zipFile, entry, entry.getName(), overWriteAll);
   }

   public boolean unpackFile(File targetDir, ZipFile zipFile, ZipEntry entry,
      String fileName, boolean overWriteAll)
     throws IOException,InterruptedException
   {
      File file = new File(targetDir, fileName);

      setInfo(getLabelWithValue("message.unpacking", file.getAbsolutePath()));

      File parent = file.getParentFile();

      if (!(parent.exists() || parent.mkdirs()))
      {
        throw new IOException(getLabelWithValue("error.io.cant_mkdir",
          parent.getAbsolutePath()));
      }

      if (entry.isDirectory())
      {
         if (!(file.exists() || file.mkdirs()))
         {
            throw new IOException(getLabelWithValue("error.io.cant_mkdir",
               file.getAbsolutePath()));
         }
      }
      else
      {
         if (overWriteAll)
         {
            Files.copy(zipFile.getInputStream(entry), file.toPath(),
               StandardCopyOption.REPLACE_EXISTING);
         }
         else if (file.exists())
         {
            // confirm overwrite

            Object thisVal = getLabel("message.overwrite.this");
            Object allVal  = getLabel("message.overwrite.all");
            Object skipVal = getLabel("message.overwrite.skip");

            Object result = JOptionPane.showInputDialog(this, 
               getLabelWithValue("message.overwrite", file.getAbsolutePath()),
               getLabel("message.confirm"),
               JOptionPane.WARNING_MESSAGE,
               null,
               new Object[]
               {
                  thisVal,
                  allVal,
                  skipVal
               },
               thisVal
            );

            if (result == null)
            {
               throw new CancelledException();
            }

            if (result == allVal)
            {
               overWriteAll = true;
            }

            if (result == allVal || result == thisVal)
            {
               Files.copy(zipFile.getInputStream(entry), file.toPath(),
                  StandardCopyOption.REPLACE_EXISTING);
            }
         }
         else
         {
            Files.copy(zipFile.getInputStream(entry), file.toPath());
         }
      }

      return overWriteAll;
   }

   public int execCommandAndWaitFor(String[] cmd)
     throws IOException,InterruptedException
   {
      return execCommandAndWaitFor(cmd, null, null, new DefaultProcessListener(this), true);
   }

   public int execCommandAndWaitFor(String[] cmd, boolean writeInfo)
     throws IOException,InterruptedException
   {
      return execCommandAndWaitFor(cmd, null, null, new DefaultProcessListener(this), writeInfo);
   }

   public int execCommandAndWaitFor(String[] cmd, ProcessListener listener)
     throws IOException,InterruptedException
   {
      return execCommandAndWaitFor(cmd, null, null, listener, true);
   }

   public int execCommandAndWaitFor(String[] cmd, String[] envp, File dir)
     throws IOException,InterruptedException
   {
      return execCommandAndWaitFor(cmd, envp, dir, new DefaultProcessListener(this), true);
   }

   public int execCommandAndWaitFor(String[] cmd, String[] envp, File dir, ProcessListener listener)
     throws IOException,InterruptedException
   {
      return execCommandAndWaitFor(cmd, envp, dir, listener, true);
   }


   // adapted from http://kylecartmell.com/?p=9
   public int execCommandAndWaitFor(String[] cmd, String[] envp, File dir, 
      ProcessListener listener, boolean writeInfo)
     throws IOException,InterruptedException
   {
      java.util.Timer timer = null;
      Process p = null;
      int exitCode = -1;

      InterruptTimerTask interruptor = null;

      try
      {
         timer = new java.util.Timer(true);
         interruptor = new InterruptTimerTask(Thread.currentThread());
         timer.schedule(interruptor, properties.getMaxProcessTime());
         listener.setInterruptor(interruptor);

         p = execCommand(cmd, envp, dir, listener, writeInfo);

         addProcessListener(listener);

         exitCode = p.waitFor();
      }
      catch (InterruptedException e)
      {
         p.destroy();

         throw e;
      }
      finally
      {
         timer.cancel();
         Thread.interrupted();
         removeProcessListener(listener);
         updateAbortItem();
      }

      return exitCode;
   }

   public Process execCommand(String[] cmd)
     throws IOException
   {
      return execCommand(cmd, null, null, new DefaultProcessListener(this), true);
   }

   public Process execCommand(String[] cmd, boolean writeInfo)
     throws IOException
   {
      return execCommand(cmd, null, null, new DefaultProcessListener(this), writeInfo);
   }

   public Process execCommand(String[] cmd, ProcessListener listener)
     throws IOException
   {
      return execCommand(cmd, null, null, listener, true);
   }

   public Process execCommand(String[] cmd, String[] envp, File dir)
     throws IOException
   {
      return execCommand(cmd, envp, dir, new DefaultProcessListener(this), true);
   }

   public Process execCommand(String[] cmd, String[] envp, File dir, ProcessListener listener)
     throws IOException
   {
      return execCommand(cmd, envp, dir, listener, true);
   }

   public Process execCommand(String[] cmd, String[] envp, File dir,
       ProcessListener listener, boolean writeInfo)
     throws IOException
   {
      String params = "";

      for (int i = 1; i < cmd.length; i++)
      {
         params += " \""+cmd[i]+"\"";
      }

      String execName = cmd[0];

      int idx = execName.lastIndexOf(File.separator);

      if (idx > -1)
      {
         execName = execName.substring(idx+1);
      }

      if (debugMode)
      {
         System.out.println("Running:");

         if (dir != null)
         {
            System.out.println("  in directory "+dir.getAbsolutePath());
         }

         if (envp != null && envp.length > 0)
         {
            System.out.println("  with environment:");

            for (int i = 0; i < envp.length; i++)
            {
               System.out.println("  "+envp[i]);
            }
         }

         System.out.println(cmd[0]+params);
      }

      if (writeInfo)
      {
         setInfo(getLabelWithValue("message.running", execName+params));
      }

      Process p = Runtime.getRuntime().exec(cmd, envp, dir);

      listener.setProcess(p);

      ProcessInputReaderThread inReaderThread 
        = new ProcessInputReaderThread(this, p, listener);
      listener.setThread(inReaderThread);
      inReaderThread.start();
      inReaderThread = null;

      return p;
   }

   public void enableTools()
   {
      resetFunctions();
      openItem.setEnabled(true);
      recentM.setEnabled(true);
      inputLabel.setEnabled(true);
      inputField.setEnabled(true);
      inputButton.setEnabled(true);
   }

   public void resetFunctions()
   {
      profileLabel.setEnabled(true);
      cmykButton.setEnabled(true);
      greyButton.setEnabled(true);
      useIccBox.setEnabled(true);
      convertButton.setEnabled(true);
      convertItem.setEnabled(true);
      outputLabel.setEnabled(true);
      outputField.setEnabled(true);
      outputButton.setEnabled(true);
      authorLabel.setEnabled(true);
      authorField.setEnabled(true);
      titleLabel.setEnabled(true);
      titleField.setEnabled(true);
      pageCountLabel.setEnabled(true);
      fileSizeLabel.setEnabled(true);
      infoBox.setEnabled(true);
   }

   public void disableTools()
   {
      disableFunctions();
      openItem.setEnabled(false);
      recentM.setEnabled(false);
      inputLabel.setEnabled(false);
      inputField.setEnabled(false);
      inputButton.setEnabled(false);
   }

   public void disableFunctions()
   {
      profileLabel.setEnabled(false);
      cmykButton.setEnabled(false);
      greyButton.setEnabled(false);
      useIccBox.setEnabled(false);
      convertButton.setEnabled(false);
      convertItem.setEnabled(false);
      outputLabel.setEnabled(false);
      outputField.setEnabled(false);
      outputButton.setEnabled(false);
      authorLabel.setEnabled(false);
      authorField.setEnabled(false);
      titleLabel.setEnabled(false);
      titleField.setEnabled(false);
      pageCountLabel.setEnabled(false);
      fileSizeLabel.setEnabled(false);
      infoBox.setEnabled(false);
   }

   public File getApp(String appName)
      throws FileNotFoundException
   {
      return getApp(appName, null);
   }

   public File getApp(String appName, String altAppName)
      throws FileNotFoundException
   {
      return getApp(appName, null, null);
   }

   public File getApp(String appName, String winAppName, String os2AppName)
      throws FileNotFoundException
   {
      File path = appSelector.fetchApplicationPath(
         appName, winAppName, os2AppName,
         getLabelWithValue("properties.query.location.app", appName));

      if (path == null || !path.exists())
      {
         String mess = getLabelWithValue("warning.no_build.missing.app", appName);

         throw new FileNotFoundException(mess);
      }

      return path;
   }

   public String getGSApp()
    throws IOException
   {
      String app = properties.getGSApp();

      if (app != null && app.length() > 0)
      {
         return app;
      }

      File path = null;

      path = getApp("gs", "gswin32c", "gsos2");

      if (path == null)
      {
         throw new FileNotFoundException(getLabel("error.no_gs"));
      }

      properties.setGSApp(path.getAbsolutePath());

      return path.getAbsolutePath();
   }


   public static void debug(String message)
   {
      if (debugMode)
      {
         System.out.println(message);
      }
   }

   public static boolean inDebugMode()
   {
      return debugMode;
   }

   public long getMaxProcessTime()
   {
      return properties.getMaxProcessTime();
   }

   // Get the filename in a form that GhostScript will accept

   public static String toFileName(File file)
   {
      String filename = file.getAbsolutePath();

      if (File.separator.equals("\\"))
      {
         filename = filename.replaceAll("\\\\", "/");
      }

      filename = filename.replaceAll("(\\\\)", "\\\\\\1")
                         .replaceAll("([\\(\\)])", "\\\\1");
      return filename;
   }

   public void getPdfInfo(File pdfFile)
     throws IOException,InterruptedException,ProcessFailedException
   {
      if (!pdfFile.exists())
      {
         throw new FileNotFoundException(getLabelWithValue("error.io.file_not_found",
            pdfFile.getAbsolutePath()));
      }

      setInfo("message.getinfo");

      String gs = getGSApp();

      String filename = toFileName(pdfFile);

      String[] cmd = new String[]
      {
         gs,
         "-dNODISPLAY",
         "-q",
         "-c",
         "("+filename+") (r) file runpdfbegin "
          + "Trailer  /Info known {Trailer /Info get} {0 packedarray} ifelse "
          + "/pdfinfo exch def "
          + "pdfinfo /Title known {pdfinfo /Title get} {(unknown)} ifelse = "
          + "pdfinfo /Author known {pdfinfo /Author get} {(unknown)} ifelse = "
          + "pdfpagecount = quit"
      };

      PdfInfoProcessListener listener = new PdfInfoProcessListener(this);

      int exitCode  = execCommandAndWaitFor(cmd, listener);

      if (exitCode != 0)
      {
         String errMess = listener.getErrorMessage();

         if (errMess != null)
         {
            errMess = getLabelWithValue("error.gsmess", errMess);
         }

         throw new ProcessFailedException(getLabelWithValues("error.noinfo", 
           new String[] {filename, errMess}));
      }

      String title = listener.getTitle();

      if (title.equals("unknown"))
      {
         title = getLabel("main", "unknown");
      }

      String author = listener.getAuthor();

      if (author.equals("unknown"))
      {
         author = getLabel("main", "unknown");
      }

      int pageCount = listener.getPageCount();

      titleField.setText(title);
      authorField.setText(author);
      pageCountField.setText(pageCount == -1 ? 
        getLabel("main", "unknown") : ""+pageCount);

      fileSizeField.setText(formatFileSize(pdfFile.length()));

      setInfo("message.done");
   }

   public static String formatFileSize(long size)
   {
      int grp = (int) (Math.log10(size)/Math.log10(1024));

      return fileSizeFormat.format(size/Math.pow(1024, grp))
        + " "+ fileSizeUnits[grp];
   }

   public static void main(String[] args)
   {
      try
      {
         for (int i = 0; i < args.length; i++)
         {
            if (args[i].equals("-debug"))
            {
               debugMode = true;
            }
            else if (args[i].equals("-nodebug"))
            {
               debugMode = false;
            }
            else if (args[i].equals("-version") || args[i].equals("--version"))
            {
               System.out.println(appName+" version "+appVersion);
               System.exit(0);
            }
            else if (args[i].equals("-help") || args[i].equals("--help"))
            {
               System.out.println(appName+" version "+appVersion);
               System.out.println("");
               System.out.println("Syntax:");
               System.out.println("");
               System.out.println("jmakepdfx [options]");
               System.out.println("");
               System.out.println("Options:");
               System.out.println("");
               System.out.println("-debug\t\t\tSwitch on debug mode");
               System.out.println("-nodebug\t\tSwitch off debug mode (default)");
               System.out.println("-version\t\tDisplay version number and exit");
               System.exit(0);
            }
            else
            {
               throw new IllegalArgumentException
                  ("Unknown option '"+args[i]+"'.");
            }
         }
      }
      catch (Exception e)
      {
         String message = appName+": "+e.getMessage()+" Use --help for help.";
         JpdfxResources.error(null, message);
         System.err.println(message);
         System.exit(1);
      }

      new Jmakepdfx();
   }

   private static final String appVersion = "0.3b";
   private static final String company = "Dickimaw Books";
   private static final String appName = "Jmakepdfx";
   private static final String appDate = "2012-10-22";

   private static Properties dictionary;

   private JToolBar toolBar;

   private HelpBroker mainHelpBroker;
   private CSH.DisplayHelpFromSource csh;

   private static boolean debugMode=false;

   private JFileChooser pdfFileChooser;

   private PdfFileFilter pdfFileFilter;

   private MenuButtonItem openItem, abortItem;

   private JMenuItem clearRecentItem, convertItem;

   private JMenu recentM;

   private ActionListener recentFilesListener;

   private MessageArea messageArea;

   private JRadioButton cmykButton, greyButton;

   private JCheckBox useIccBox;

   private JButton convertButton, inputButton, outputButton;

   private JLabel inputLabel, outputLabel, profileLabel,
     authorLabel, titleLabel, pageCountLabel, fileSizeLabel;

   private JTextField inputField, outputField, authorField,
      titleField, pageCountField, fileSizeField;

   private Box infoBox;

   private JpdfxProperties properties;

   private JpdfxAppSelector appSelector;

   private IccSelector iccSelector;

   private PropertiesDialog propertiesDialog;

   private Vector<Thread> currentWorkerThreads;

   private Vector<ProcessListener> currentProcessListeners;

   private File currentFile;

   private LicenceDialog licenceDialog;

   protected static DecimalFormat fileSizeFormat = new DecimalFormat("#,##0.#");

   private static final String[] fileSizeUnits
      = new String[] {"B", "KB", "MB", "GB", "TB"};
}

class MessageArea extends JLabel implements Runnable
{
   public MessageArea(String initialValue)
   {
      super(initialValue);
      message = "";
   }

   public void run()
   {
      setText(message);
   }

   public void update(String message)
   {
      this.message = message;

      Thread thread = new Thread(this);

      thread.start();
      thread = null;
   }

   private String message;
}
